(*

Call Method Oriented Project.applescript

Copyright 2007 ppm. All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

	Redistributions of source code must retain the above copyright notice,
	this list of conditions and the following disclaimer.

	Redistributions in binary form must reproduce the above copyright notice,
	this list of conditions and the following disclaimer in the documentation
	and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY 
EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*)



(* ################ Properties ################ *)


(* %%%%%%%% Shared Script Objects %%%%%%%% *)

property controller : missing value
property URLsController : missing value
property webView : missing value


(* %%%%%%%% Outlets Holder %%%%%%%% *)

property outlets : missing value








(* ################ Event Handlers ################ *)


on awake from nib theObject
	
	set objectName to name of theObject
	
	-- Below registers references of all objects, whose awakeFromNib handler is enabled on IB.
	-- And you can retrive the object with function outlet().
	-- In other words, no more : 'text view "description" of scroll view "scroller" of box "descriptionBox" of tab view item "page1" of tab view "pageTabView" of window "main"'.
	-- Just: outlet("description")
	
	if outlets is missing value then set outlets to dictionary() of SO_Dictionary()
	addOutlet(theObject, objectName)
	
	-- Do real awakeFromNib initialization here.
	
	-- if objectName is "" then
	-- end if
	
end awake from nib


on launched theObject
	
	-- Set up shared script objects.
	
	set controller to object() of SO_Controller()
	
	-- NSArrayController can't be retrived through 'outlet' system above,
	-- So get AppController's IBOutlet with KVC.
	
	set URLsController to wrapperWithObject(t_vfk("URLsController")) of SO_ArrayControllerWrapper()
	set webView to wrapperWithWebView(outlet("webView")) of SO_WebViewWrapper()
	
end launched


-- I rarely write actual logic within event handler.
-- Here is place for just send message to some object.
-- This would be useful like when you have to bind same action to menu item.
-- Easy to manage.
on clicked theObject
	
	set objectName to name of theObject
	
	if objectName is "addURL" then
		tell controller to addURL()
		
	else if objectName is "removeURL" then
		tell controller to removeURL()
		
	else if objectName is "openURL" then
		tell controller to openURL()
		
	end if
	
end clicked








(* ################ Custom Implementations ################ *)


on SO_Controller()
	
	script |Controller|
		
		property parent : SO_Object()
		
		on addURL()
			
			set URLString to string value of outlet("URLField")
			
			if URLString is not "" then
				
				set dict to dictionary() of SO_Dictionary()
				
				tell dict to setObjectforKey(URLString, "URL")
				tell URLsController to addObject(wrappedObject of dict) -- Of course, wrapper object itself cannot be used as dictionary.
				
			end if
			
		end addURL
		
		on removeURL()
			
			tell URLsController to if canRemove() then |remove|()
			
		end removeURL
		
		on openURL()
			
			set URLs to selectedObjects() of URLsController
			
			if number of URLs is greater than 0 then
				
				set dict to dictionaryWithDictionary(first item of URLs) of SO_Dictionary()
				
				tell webView to displayURL(objectForKey("URL") of dict)
				
			end if
			
		end openURL
		
	end script
	
end SO_Controller








(* ################ Functions ################ *)

(* %%%%%%%% Essential %%%%%%%%% *)

(* ==== Core Call Method Shortcut Functions ==== *)

-- Raw 'call method' syntax is too hard to type again and again. So these are shortcuts for that.
-- Also, one problem of 'call method' is when it returned nil. AppleScript is nothing to do with it,
-- and causes error *next time* you attempt to use variable contains nil. Not just after it returned nil.
-- So these functions always returns familier 'missing value', if returned value was nil.


-- (C)alls (m)ethod (o)f some (o)bject, without any parameters. 
on cmoo(givenMethod, givenObject)
	
	if givenObject is not missing value then
		
		set returnValue to call method givenMethod of givenObject
		
		try
			return returnValue
		end try
		
	end if
	
	return missing value
	
end cmoo


-- (C)alls (m)ethod (o)f some (o)bject, (w)ith single (p)arameter.
-- In other words, this function is for Cocoa method which have single ':'.
on cmoowp(givenMethod, givenObject, givenParameter)
	
	if givenObject is not missing value then
		
		set returnValue to call method givenMethod of givenObject with parameter givenParameter
		
		try
			return returnValue
		end try
		
	end if
	
	return missing value
	
end cmoowp


-- (C)alls (m)ethod (o)f some (o)bject, (w)ith multiple (p)arameter(s).
-- In other words, this function is for Cocoa method which have multiple ':'s, at least two.
on cmoowps(givenMethod, givenObject, givenParameters)
	
	if givenObject is not missing value then
		
		set returnValue to call method givenMethod of givenObject with parameters givenParameters
		
		try
			return returnValue
		end try
		
	end if
	
	return missing value
	
end cmoowps


on cmoc(givenMethod, givenClass)
	
	set returnValue to call method givenMethod of class givenClass
	
	try
		return returnValue
	on error
		return missing value
	end try
	
end cmoc


on cmocwp(givenMethod, givenClass, givenParameter)
	
	set returnValue to call method givenMethod of class givenClass with parameter givenParameter
	
	try
		return returnValue
	on error
		return missing value
	end try
	
end cmocwp


on cmocwps(givenMethod, givenClass, givenParameters)
	
	set returnValue to call method givenMethod of class givenClass with parameters givenParameters
	
	try
		return returnValue
	on error
		return missing value
	end try
	
end cmocwps



(* ==== Shortcuts For Call Method Of 'Top Level Object' ==== *)

-- If you 'call method' without providing target object, it means call method of 'top level object'.
-- 'Top level object' can be NSApp and [NSApp delegate].
-- So this functions are useful if you created support method for your script in NSApplication category or delegate of it.


on t_cm(givenMethod)
	
	set returnValue to call method givenMethod
	
	try
		return returnValue
	on error
		return missing value
	end try
	
end t_cm


on t_cmwp(givenMethod, givenParameter)
	
	set returnValue to (call method givenMethod with parameter givenParameter)
	
	try
		return returnValue
	on error
		return missing value
	end try
	
end t_cmwp


on t_cmwps(givenMethod, givenParameters)
	
	set returnValue to call method givenMethod with parameters givenParameters
	
	try
		return returnValue
	on error
		return missing value
	end try
	
end t_cmwps



(* ==== Shortcuts for Accessing Properties of 'Top Level Object' with KVC ==== *)

-- These are useful for get/set properties (instance variables) of AppDelegate.
-- Say, you have 'myArrayController' outlet in AppDelegate,
-- then you can get it with t_vfk("myArrayController")


on t_vfk(givenKey)
	return t_cmwp("valueForKey:", givenKey)
end t_vfk


-- Utility function guarantees return value is boolean.
on t_bfk(givenKey)
	return booleanValue(t_vfk(givenKey))
end t_bfk


on t_svfk(givenValue, givenKey)
	return t_cmwps("setValue:forKey:", {givenValue, givenKey})
end t_svfk



(* ==== Shortcuts For Accessing Properties Of Some Object With KVC ==== *)

on vfkoo(givenKey, givenObject)
	
	set returnValue to cmoowp("valueForKey:", givenObject, givenKey)
	
	try
		return returnValue
	on error
		return missing value
	end try
	
end vfkoo


on vfkpoo(givenKeyPath, givenObject)
	
	set returnValue to cmoowp("valueForKeyPath:", givenObject, givenKeyPath)
	
	try
		return returnValue
	on error
		return missing value
	end try
	
end vfkpoo


on svfkoo(givenValue, givenKey, givenObject)
	return cmoowps("setValue:forKey:", givenObject, {givenValue, givenKey})
end svfkoo


on svfkpoo(givenValue, givenKeyPath, givenObject)
	return cmoowps("setValue:forKeyPath:", givenObject, {givenValue, givenKeyPath})
end svfkpoo



(* ==== Forcing Value To Be Some Useful Value ==== *)

-- These are mainly for use in conjunction with above call method shortcuts, but maybe are usable with other values.
-- Use these in case you want to get empty string, rather than missing value, when returned value was missing value.


on stringValue(givenObject)
	
	if givenObject is not missing value then
		try
			return givenObject as string
		end try
	end if
	
	return ""
	
end stringValue


on listValue(givenObject)
	
	if givenObject is not missing value then
		try
			return givenObject as list
		end try
	end if
	
	return {}
	
end listValue


on booleanValue(givenObject)
	
	if givenObject is not missing value then
		try
			return givenObject as boolean
		end try
	end if
	
	return false
	
end booleanValue


on integerValue(givenObject)
	
	if givenObject is not missing value then
		try
			return givenObject as integer
		end try
	end if
	
	return 0
	
end integerValue




(* ==== Shortcuts For Displaying String ==== *)


on lso(str)
	return localized string of str
end lso


on qfo(str)
	return quoted form of str
end qfo


on dqfo(str)
	return (data utxt0022 as class utf8) & (str as class utf8) & (data utxt0022 as class utf8)
end dqfo




(* ==== String Manipulation ==== *)


on split(givenSeparator, givenString)
	return cmoowp("componentsSeparatedByString:", givenString, givenSeparator)
end split


on join(givenSeparator, givenArray)
	return cmoowp("componentsJoinedByString:", givenArray, givenSeparator)
end join


on replace(givenStringToFind, givenStringToReplace, givenSourceString)
	return join(givenStringToReplace, split(givenStringToFind, givenSourceString))
end replace




(* ==== Shortcuts For Outlet Management ==== *)


on addOutlet(givenOutlet, givenName)
	tell outlets to setObjectforKey(givenOutlet, givenName)
	log ("addOutlet() -- Added: " & class of givenOutlet as string) & space & qfo(givenName)
end addOutlet


on outlet(givenName)
	return objectForKey(givenName) of outlets
end outlet








(* ################ Script Objects ################ *)


(* %%%%%%%% Essential %%%%%%%% *)


-- Root object.
-- If you want to modify every script objects' behavior, customize this.
on SO_Object()
	
	script |Object|
		
		on object()
			return me
		end object
		
	end script
	
end SO_Object


-- Wrapper object for some object, including ones not reachable with raw AppleScript.
-- It can wrap an object and have methods (handlers) to call 'call method' on it.
-- So this is useful to subclass(?), if you want to add custom behavior to the object,
-- rather than to simply call 'call method' on the object.
on SO_Wrapper()
	
	script |Wrapper|
		
		property parent : SO_Object()
		property wrappedObject : missing value
		
		on wrapperWithObject(givenObject)
			set wrappedObject to givenObject
			continue object()
		end wrapperWithObject
		
		on cm(givenMethod)
			return cmoo(givenMethod, wrappedObject)
		end cm
		
		on cmwp(givenMethod, givenParameter)
			return cmoowp(givenMethod, wrappedObject, givenParameter)
		end cmwp
		
		on cmwps(givenMethod, givenParameters)
			return cmoowps(givenMethod, wrappedObject, givenParameters)
		end cmwps
		
	end script
	
end SO_Wrapper


-- (Virtually mutable) dictionary.
-- Good example of how to subclass Wrapper. And perhaps one of the first objects you want in AppleScript.
on SO_Dictionary()
	
	script |Dictionary|
		
		property parent : SO_Wrapper()
		
		on dictionary()
			continue wrapperWithObject(cmoc("dictionary", "NSDictionary"))
		end dictionary
		
		on dictionaryWithDictionary(givenDictionary)
			
			-- 'givenDictionary' accepts AppleScript record.
			-- So you can make record usable like NSDictionary.
			
			continue wrapperWithObject(givenDictionary)
			
		end dictionaryWithDictionary
		
		on objectForKey(givenKey)
			return cmwp("objectForKey:", givenKey)
		end objectForKey
		
		on setObjectforKey(givenObject, givenKey)
			
			-- if givenKey is not in wrappedObject(NSDictionary instance), this adds the key with value.
			-- if givenKey already exists, this updates the key's value.
			
			set dict to cmocwps("dictionaryWithObject:forKey:", "NSDictionary", {givenObject, givenKey})
			
			-- Use NSDictionary instance as AppleScript record.
			-- Those are automatically converted between two.
			tell (a reference to my wrappedObject) to set contents to dict & contents
			
		end setObjectforKey
		
	end script
	
end SO_Dictionary




(* %%%%%%%% Additional %%%%%%%% *)


-- NSArrayController wrapper.
on SO_ArrayControllerWrapper()
	
	script |ArrayControllerWrapper|
		
		property parent : SO_Wrapper()
		
		-- setValue:forKeyPath: shortcut.
		on svfkp(givenValue, givenKeyPath)
			return cmwps("setValue:forKeyPath:", {givenValue, givenKeyPath})
		end svfkp
		
		-- value:forKeyPath: shortcut.
		on vfkp(givenKeyPath)
			return cmwp("valueForKeyPath:", givenKeyPath)
		end vfkp
		
		-- Below three methods are not in Cocoa API.
		-- Forces to return some useful value, not missing value.
		on sfkp(givenKeyPath)
			return stringValue(vfkp(givenKeyPath))
		end sfkp
		
		on lfkp(givenKeyPath)
			return listValue(vfkp(givenKeyPath))
		end lfkp
		
		on bfkp(givenKeyPath)
			return booleanValue(vfkp(givenKeyPath))
		end bfkp
		
		on addObject(givenObject)
			return cmwp("addObject:", givenObject)
		end addObject
		
		on insertObjectatArrangedObjectIndex(givenObject, givenIndex)
			return cmwps("insertObject:atArrangedObjectIndex:", {givenObject, givenIndex})
		end insertObjectatArrangedObjectIndex
		
		on setSelectionIndex(givenIndex)
			return cmwp("setSelectionIndex:", givenIndex)
		end setSelectionIndex
		
		on removeObjectAtArrangedObjectIndex(givenIndex)
			return cmwp("removeObjectAtArrangedObjectIndex:", givenIndex)
		end removeObjectAtArrangedObjectIndex
		
		on setContent(givenObject)
			return cmwp("setContent:", givenObject)
		end setContent
		
		on canRemove()
			return booleanValue(cm("canRemove"))
		end canRemove
		
		on |remove|()
			cmwp("remove:", null)
		end |remove|
		
		on selectedObjects()
			return cm("selectedObjects")
		end selectedObjects
		
	end script
	
end SO_ArrayControllerWrapper


-- WebView Wrapper
on SO_WebViewWrapper()
	
	script |WebViewWrapper|
		
		property parent : SO_Wrapper()
		property mainFrame : missing value
		
		on wrapperWithWebView(givenWebView)
			
			set aWebView to continue wrapperWithObject(givenWebView)
			set mainFrame of aWebView to cm("mainFrame")
			
			return aWebView
			
		end wrapperWithWebView
		
		on displayURL(givenURL)
			
			set anURL to NSURLize(givenURL)
			set request to cmocwp("requestWithURL:", "NSURLRequest", anURL)
			
			cmoowp("loadRequest:", mainFrame, request)
			
		end displayURL
		
		on displayHTML(givenHTML)
			cm("loadHTMLString:baseURL:", mainFrame, {givenHTML, NSURLize("")})
		end displayHTML
		
		on NSURLize(givenURLString)
			return cmocwp("URLWithString:", "NSURL", givenURLString)
		end NSURLize
		
		on waitUntilProcessFinish()
			repeat
				if cm("estimatedProgress") is 0 then exit repeat
			end repeat
		end waitUntilProcessFinish
		
	end script
	
end SO_WebViewWrapper







